//----------------------------------------------------------------------------//
//- CapEtikettenReport.pas ---------------------------------------------------//
//----------------------------------------------------------------------------//

//- Beschreibung -------------------------------------------------------------//
// Wird zum Drucken der Preisetiketten bentigt                               //
//----------------------------------------------------------------------------//

unit CapEtikettenReport;

//------------------------------------------------------------------------------

interface

//------------------------------------------------------------------------------

uses
  Windows, SysUtils, Messages, Classes, Graphics, Controls, StdCtrls, ExtCtrls,
  Forms, QuickRpt, QRCtrls, CaoArtikelListe, CaoFormat, CaoRegistry;

//------------------------------------------------------------------------------

type
  TCapQrEtikettenReport = class(TQuickRep)
    m_QRBndBarecode      : TQRStringsBand;
    m_QrLblBarecode1     : TQRLabel;
    m_QrLblBeschreibung1 : TQRLabel;
    m_QrLblPreis1        : TQRLabel;
    m_QrLblBarecode2     : TQRLabel;
    m_QrLblBeschreibung2 : TQRLabel;
    m_QrLblPreis2        : TQRLabel;
    m_QrLblBarecode3     : TQRLabel;
    m_QrLblBeschreibung3 : TQRLabel;
    m_QrLblPreis3        : TQRLabel;

    //--------------------------------------------------------------------------

    procedure QuickRepBeforePrint     (Sender         : TCustomQuickRep;
                                       var PrintReport: Boolean);
    procedure QRBndBarecodeBeforePrint(Sender         : TQRCustomBand;
                                       var PrintBand  : Boolean);
    procedure QuickRepAfterPrint(Sender: TObject);

  //----------------------------------------------------------------------------

  private
    m_obArtikelListe    : TCaoArtikelListe;
    m_obFormat          : TCaoFormat;
    m_obRegistry        : TCaoRegistry;

    //--------------------------------------------------------------------------

    procedure PrvFillBand  (obArtikel      : TCaoArtikel;
                            iPos           : Integer);
    procedure PrvFillLabels(obArtikel      : TCaoArtikel;
                            LblBeschreibung: TQrLabel;
                            LblPreis       : TQrLabel;
                            LblBarecode    : TQrLabel);
    procedure PrvClearBand;
    procedure PrvClearLabels       (iPos: Integer);
    procedure PrvCreateArtikelListe(obArtikelListe: TCaoArtikelListe;
                                    iAnzEtiketten : Integer);

    
  //----------------------------------------------------------------------------

  public
    procedure Print  (obArtikelListe    : TCaoArtikelListe;
                      iFehlendeEtiketten: Integer); reintroduce;
    procedure Preview(obArtikelListe    : TCaoArtikelListe;
                      iFehlendeEtiketten: Integer); reintroduce;

  //----------------------------------------------------------------------------

  end; // TCapQrEtikettenReport

//------------------------------------------------------------------------------

var
  CapQrEtikettenReport: TCapQrEtikettenReport;

//------------------------------------------------------------------------------

implementation

//------------------------------------------------------------------------------

uses
  CaoPersonenListe;

//------------------------------------------------------------------------------

{$R *.DFM}

//------------------------------------------------------------------------------

procedure TCapQrEtikettenReport.Print(obArtikelListe    : TCaoArtikelListe;
                                      iFehlendeEtiketten: Integer); 
begin
  PrvCreateArtikelListe(obArtikelListe, iFehlendeEtiketten);
  inherited Print;
end; // TCapQrEtikettenReport.Print

//------------------------------------------------------------------------------

procedure TCapQrEtikettenReport.Preview(obArtikelListe    : TCaoArtikelListe;
                                        iFehlendeEtiketten: Integer);
begin
  PrvCreateArtikelListe(obArtikelListe, iFehlendeEtiketten);
  inherited Preview;
end; // TCapQrEtikettenReport.Preview

//------------------------------------------------------------------------------

procedure TCapQrEtikettenReport.QuickRepBeforePrint(Sender: TCustomQuickRep;
                                                    var PrintReport: Boolean);
var
  iIndex: Integer;
begin
  m_obRegistry := TCaoRegistry.Create;
  m_QRBndBarecode.Items.Clear;
  // Die bentigte Anzahl Bnder erzeugen
  // Bentigt werden die Anzahl Etiketten, die ausgedruckt werden sollen / 3
  // da in jedes Band drei Etiketten passen
  if m_obArtikelListe.Count > 0 then begin
    m_QRBndBarecode.Items.Add(IntToStr(0));
    for iIndex := 0 to (m_obArtikelListe.Count div 3) -1 do begin
      m_QRBndBarecode.Items.Add(IntToStr(iIndex + 1));
    end; // for
  end; // if
end; // TCapQrEtikettenReport.QuickRepBeforePrint

//------------------------------------------------------------------------------

procedure TCapQrEtikettenReport.QRBndBarecodeBeforePrint(Sender: TQRCustomBand;
                                                         var PrintBand: Boolean);
var
  iBandNr, iIndex: Integer;
begin
  // Auslesen der Bandnummer, sie wurde in QuickRepBeforePrint vergeben
  iBandNr := StrToInt(m_QRBndBarecode.Item);
  if iBandNr >= 0 then begin
    for iIndex := 0 to 2 do begin
      if (iBandNr * 3 + iIndex) <= m_obArtikelListe.Count -1 then begin
        PrvFillBand(m_obArtikelListe.Items[iBandNr * 3 + iIndex], iIndex);
      end // if
      else begin
        PrvClearLabels(iIndex);
      end; // else 
    end; // for
  end // if
  else begin
    PrvClearBand;
  end; // else
end; // TCapQrEtikettenReport.QRBndBarecodeBeforePrint

//------------------------------------------------------------------------------

procedure TCapQrEtikettenReport.PrvFillBand(obArtikel: TCaoArtikel;
                                            iPos     : Integer);
(*
  Abfllen eines Qr-Bandes mit den Artikeldaten
*)
begin
  case iPos of
    0: PrvFillLabels(obArtikel, m_QrLblBeschreibung1, m_QrLblPreis1, m_QrLblBarecode1);
    1: PrvFillLabels(obArtikel, m_QrLblBeschreibung2, m_QrLblPreis2, m_QrLblBarecode2);
    2: PrvFillLabels(obArtikel, m_QrLblBeschreibung3, m_QrLblPreis3, m_QrLblBarecode3);
  end; // case
end; // TCapQrEtikettenReport.PrvFillBand

//------------------------------------------------------------------------------

procedure TCapQrEtikettenReport.PrvFillLabels(obArtikel: TCaoArtikel;
                                              LblBeschreibung,
                                              LblPreis,
                                              LblBarecode: TQrLabel);
(*
  Abfllen der ArtikelLabels
*)
var
  sBeschreibung: string;
begin
  // Besitzt der Artikel eine Artikelnummer, so werden die Labels mit
  // den Artikeldaten abgefllt
  if obArtikel.iArtikelNummer > 0 then begin
    sBeschreibung           := obArtikel.sBeschreibung;
    // Je nach Einstellung werden die Initialen auf die Etiketten gedruckt
    // oder nicht
    if m_obRegistry.ReadInitialenDrucken then begin
      sBeschreibung := sBeschreibung + ' (' +
                       TCaoPersonen(obArtikel.pAnbieter).sInitialen + ')';
    end; // if
    LblBeschreibung.Caption := sBeschreibung;
    LblPreis.Caption        := m_obFormat.FormatCurrency(obArtikel.dPreis);
    LblBarecode.Caption     := m_obFormat.FormatBarCode(obArtikel.iArtikelNummer);
  end // if
  // Ist die Artikelnummer 0, dann werden die Labels mit '' abgefllt
  else begin
    LblBeschreibung.Caption := '';
    LblPreis.Caption        := '';
    LblBarecode.Caption     := '';
  end; // else
end; // TCapQrEtikettenReport.PrvFillLabels

//------------------------------------------------------------------------------

procedure TCapQrEtikettenReport.PrvClearBand;
(*
  Leeren aller Inhalte der Labels auf dem Stringband
*)
var
  iIndex: Integer;
begin
  for iIndex := 0 to 2 do begin
    PrvClearLabels(iIndex);
  end; // for
end; // TCapQrEtikettenReport.PrvClearBand

//------------------------------------------------------------------------------

procedure TCapQrEtikettenReport.PrvClearLabels(iPos: Integer);
(*
  Die Labels mit der Beschreibung, dem Preis und dem Barcode an der angegebenen
  Position lschen
*)
begin
  case iPos of
    0:
      begin
        m_QrLblBeschreibung1.Caption := '';
        m_QrLblPreis1.caption        := '';
        m_QrLblBarecode1.Caption     := '';
      end; // case 0
    1:
      begin
        m_QrLblBeschreibung2.Caption := '';
        m_QrLblPreis2.caption        := '';
        m_QrLblBarecode2.Caption     := '';
      end; // case 1
    2:
      begin
        m_QrLblBeschreibung3.Caption := '';
        m_QrLblPreis3.caption        := '';
        m_QrLblBarecode3.Caption     := '';
      end; // case 2
  end; // case
end; // TCapQrEtikettenReport.PrvClearLabels

//------------------------------------------------------------------------------

procedure TCapQrEtikettenReport.PrvCreateArtikelListe(obArtikelListe: TCaoArtikelListe;
                                                      iAnzEtiketten : Integer);
(*
  Legt eine Temporre Artikelliste an und fllt diese mit iAnzEtiketten
  leeren Artikelobjekten auf. Anschliessend wird die Artikelliste mit der
  Anzahl zu druckenden Etiketten in die Temporre ArtikelListe kopiert
*)
var
  iIndex   : Integer;
  obArtikel: TCaoArtikel;
begin
  m_obArtikelListe := TCaoArtikelListe.Create;
  // Wenn in einem Band einzelne Etiketten fehlen, mssen diese durch Artikel-
  // Objekte ohne Daten abgefllt werden
  for iIndex := 0 to iAnzEtiketten -1 do begin
    obArtikel := TCaoArtikel.Create;
    m_obArtikelListe.AddArtikel(obArtikel);
  end; // for  
  // Die gesamte Artikelliste wird in ein neues Objekt kopiert, so kann mit
  // der neuen Artikelliste gearbeitet werden, ohne an den ursprnglichen Daten
  // etwas zu verndern
  m_obArtikelListe.CopyArtikelListe(obArtikelListe);
end; // TCapQrEtikettenReport.PrvCreateArtikelListe

//------------------------------------------------------------------------------

procedure TCapQrEtikettenReport.QuickRepAfterPrint(Sender: TObject);
begin
  m_obRegistry.Free;
  m_obArtikelListe.Free(True);
end; // TCapQrEtikettenReport.QuickRepAfterPrint

//------------------------------------------------------------------------------

end. // CapEtikettenReport
